<?php

namespace App\Http\Controllers;

use App\Models\InquiryRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Exception;

class InquiryRequestController extends Controller
{
    // 🔹 LIST (with User)
    public function index()
    {
        try {
            $data = InquiryRequest::with('user')
                ->orderBy('Inq_Id', 'desc')
                ->get();

            return response()->json([
                'status' => 'success',
                'data' => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'U_Id'     => 'required|exists:users_masters,U_Id',
                'Inq_Date' => 'required|date',
                'Inq_Desc' => 'required|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = InquiryRequest::create([
                'U_Id'     => $request->U_Id,
                'Inq_Date' => $request->Inq_Date,
                'Inq_Desc' => $request->Inq_Desc
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Enquiry created successfully',
                'data' => $data->load('user')
            ], 201);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW (single)
    public function show($id)
    {
        try {
            $data = InquiryRequest::with('user')->find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => $data
            ]);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $data = InquiryRequest::find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'U_Id'     => 'nullable|exists:users_masters,U_Id',
                'Inq_Date' => 'required|date',
                'Inq_Desc' => 'required|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data->update([
                'U_Id'     => $request->U_Id ?? $data->U_Id,
                'Inq_Date' => $request->Inq_Date,
                'Inq_Desc' => $request->Inq_Desc
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Enquiry updated successfully',
                'data' => $data->load('user')
            ]);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $data = InquiryRequest::find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Record not found'
                ], 404);
            }

            $data->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Enquiry deleted successfully'
            ]);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
