<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Organisation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Exception;

class OrganisationController extends Controller
{
    // 🔹 Get All Organisations
    public function index()
    {
        try {
            $orgs = Organisation::all()->map(function ($org) {
                $org->Org_Logo = $org->Org_Logo
                    ? asset('storage/' . $org->Org_Logo)
                    : null;
                return $org;
            });

            return response()->json([
                'status' => true,
                'message' => 'Organisation list fetched successfully',
                'data' => $orgs
            ], 200);

        } catch (Exception $e) {
            Log::error('Organisation index error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Create Organisation
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Org_Name' => 'required|string|max:255',
                'Org_Code' => 'required|string|max:50|unique:organisations,Org_Code',
                'Org_Logo' => 'nullable'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $logoPath = null;

            if ($request->hasFile('Org_Logo') || $request->Org_Logo) {
                $logoPath = $this->uploadLogo(
                    $request->file('Org_Logo') ?? $request->Org_Logo
                );
            }

            $org = Organisation::create([
                'Org_Name' => $request->Org_Name,
                'Org_Code' => $request->Org_Code,
                'Org_GST_No' => $request->Org_GST_No,
                'Org_TDS_No' => $request->Org_TDS_No,
                'Org_PAN_No' => $request->Org_PAN_No,
                'Org_Address' => $request->Org_Address,
                'Org_City' => $request->Org_City,
                'Org_State' => $request->Org_State,
                'Org_Pin' => $request->Org_Pin,
                'Org_Email' => $request->Org_Email,
                'Org_Mobile' => $request->Org_Mobile,
                'Org_URL' => $request->Org_URL,
                'Org_Logo' => $logoPath
            ]);

            $org->Org_Logo = $logoPath ? asset('storage/' . $logoPath) : null;

            return response()->json([
                'status' => true,
                'message' => 'Organisation created successfully',
                'data' => $org
            ], 201);

        } catch (Exception $e) {
            Log::error('Organisation store error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to create organisation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Update Organisation
    public function update(Request $request, $id)
    {
        try {
            $org = Organisation::find($id);

            if (!$org) {
                return response()->json([
                    'status' => false,
                    'message' => 'Organisation not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Org_Name' => 'required|string|max:255',
                'Org_Code' => 'required|string|max:50|unique:organisations,Org_Code,' . $id,
                'Org_Logo' => 'nullable'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $logoPath = $org->Org_Logo;

            if ($request->hasFile('Org_Logo') || $request->Org_Logo) {
                $logoPath = $this->uploadLogo(
                    $request->file('Org_Logo') ?? $request->Org_Logo,
                    $org->Org_Logo
                );
            }

            $org->update([
                'Org_Name' => $request->Org_Name,
                'Org_Code' => $request->Org_Code,
                'Org_GST_No' => $request->Org_GST_No,
                'Org_TDS_No' => $request->Org_TDS_No,
                'Org_PAN_No' => $request->Org_PAN_No,
                'Org_Address' => $request->Org_Address,
                'Org_City' => $request->Org_City,
                'Org_State' => $request->Org_State,
                'Org_Pin' => $request->Org_Pin,
                'Org_Email' => $request->Org_Email,
                'Org_Mobile' => $request->Org_Mobile,
                'Org_URL' => $request->Org_URL,
                'Org_Logo' => $logoPath
            ]);

            $org->Org_Logo = $logoPath ? asset('storage/' . $logoPath) : null;

            return response()->json([
                'status' => true,
                'message' => 'Organisation updated successfully',
                'data' => $org
            ], 200);

        } catch (Exception $e) {
            Log::error('Organisation update error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to update organisation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Delete Organisation
    public function destroy($id)
    {
        try {
            $org = Organisation::find($id);

            if (!$org) {
                return response()->json([
                    'status' => false,
                    'message' => 'Organisation not found'
                ], 404);
            }

            if ($org->Org_Logo && Storage::disk('public')->exists($org->Org_Logo)) {
                Storage::disk('public')->delete($org->Org_Logo);
            }

            $org->delete();

            return response()->json([
                'status' => true,
                'message' => 'Organisation deleted successfully'
            ], 200);

        } catch (Exception $e) {
            Log::error('Organisation delete error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete organisation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Logo Upload Helper (File + Base64)
    private function uploadLogo($logo, $oldLogo = null)
    {
        if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
            Storage::disk('public')->delete($oldLogo);
        }

        // FILE upload
        if ($logo instanceof \Illuminate\Http\UploadedFile) {
            return $logo->store('org_logos', 'public');
        }

        // BASE64 upload
        if (preg_match('/^data:image\/(\w+);base64,/', $logo, $type)) {
            $data = substr($logo, strpos($logo, ',') + 1);
            $data = base64_decode($data);

            $fileName = 'org_logos/' . uniqid() . '.' . $type[1];
            Storage::disk('public')->put($fileName, $data);

            return $fileName;
        }

        return null;
    }
}
