<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\RoleMapping;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Exception;

class RoleMappingController extends Controller
{
    // 🔹 Get All Role Mappings
    public function index()
    {
        try {
            $data = RoleMapping::with(['user', 'role'])->get();

            return response()->json([
                'status' => 'success',
                'data' => $data
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Create Role Mapping
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'U_Id' => 'required|exists:users_masters,U_Id',
                'Role_Id' => 'required|exists:role_masters,Role_Id',
                'Default_Rid' => 'nullable|integer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Ensure only ONE default role per user
            if ($request->Default_Rid == 1) {
                RoleMapping::where('U_Id', $request->U_Id)
                    ->update(['Default_Rid' => 0]);
            }

            $mapping = RoleMapping::create([
                'U_Id' => $request->U_Id,
                'Role_Id' => $request->Role_Id,
                'Default_Rid' => $request->Default_Rid ?? 0
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Role mapped successfully',
                'data' => $mapping
            ], 201);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Get Single Role Mapping
    public function show($id)
    {
        try {
            $mapping = RoleMapping::with(['user', 'role'])->find($id);

            if (!$mapping) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Role mapping not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => $mapping
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Update Role Mapping
    public function update(Request $request, $id)
    {
        try {
            $mapping = RoleMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Role mapping not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'U_Id' => 'exists:users_masters,U_Id',
                'Role_Id' => 'exists:role_masters,Role_Id',
                'Default_Rid' => 'integer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            if ($request->Default_Rid == 1) {
                RoleMapping::where('U_Id', $mapping->U_Id)
                    ->update(['Default_Rid' => 0]);
            }

            $mapping->update($request->all());

            return response()->json([
                'status' => 'success',
                'message' => 'Role mapping updated successfully',
                'data' => $mapping
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Delete Role Mapping
    public function destroy($id)
    {
        try {
            $mapping = RoleMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Role mapping not found'
                ], 404);
            }

            $mapping->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Role mapping deleted successfully'
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
