<?php

namespace App\Http\Controllers;

use App\Mail\OtpMail;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Support\Facades\Mail;

class UserMasterController extends Controller
{
    // 🔹 Get All Users with Role
    public function index()
    {
        try {
            $users = UserMaster::with('role')->get();

            return response()->json([
                'status' => 'success',
                'data' => $users
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Create User
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Role_Id'    => 'required|exists:role_masters,Role_Id',
                'U_Name'     => 'required|max:255',
                'U_Email'    => 'required|email|unique:users_masters,U_Email',
                'U_Password' => 'required|min:6',
                'U_Mobile'   => 'nullable|digits:10'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $user = UserMaster::create([
                'Role_Id'    => $request->Role_Id,
                'U_Name'     => $request->U_Name,
                'U_Email'    => $request->U_Email,
                'U_Password' => Hash::make($request->U_Password),
                'U_Status'   => $request->U_Status ?? '1',
                'U_Gender'   => $request->U_Gender,
                'U_Mobile'   => $request->U_Mobile,
                'U_DOB'      => $request->U_DOB,
                'U_Image'    => $request->U_Image,
                'FCM_Token'  => $request->FCM_Token
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'User created successfully',
                'data' => $user->load('role')
            ], 201);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Get Single User
    public function show($id)
    {
        try {
            $user = UserMaster::with('role')->find($id);

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => $user
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Update User
    public function update(Request $request, $id)
    {
        try {
            $user = UserMaster::find($id);

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Role_Id'  => 'nullable|exists:role_masters,Role_Id',
                'U_Email'  => 'email|unique:users_masters,U_Email,' . $id . ',U_Id',
                'U_Mobile' => 'nullable|digits:10'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $user->update([
                'Role_Id'    => $request->Role_Id ?? $user->Role_Id,
                'U_Name'     => $request->U_Name ?? $user->U_Name,
                'U_Email'    => $request->U_Email ?? $user->U_Email,
                'U_Password' => $request->U_Password
                    ? Hash::make($request->U_Password)
                    : $user->U_Password,
                'U_Status'   => $request->U_Status ?? $user->U_Status,
                'U_Gender'   => $request->U_Gender ?? $user->U_Gender,
                'U_Mobile'   => $request->U_Mobile ?? $user->U_Mobile,
                'U_DOB'      => $request->U_DOB ?? $user->U_DOB,
                'U_Image'    => $request->U_Image ?? $user->U_Image,
                'FCM_Token'  => $request->FCM_Token ?? $user->FCM_Token
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'User updated successfully',
                'data' => $user->load('role')
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Delete User
    public function destroy($id)
    {
        try {
            $user = UserMaster::find($id);

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found'
                ], 404);
            }

            $user->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'User deleted successfully'
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

public function login(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'login'    => 'required', // email OR mobile
            'password' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'validation_error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = UserMaster::where('U_Email', $request->login)
            ->orWhere('U_Mobile', $request->login)
            ->first();

        if (!$user || !Hash::check($request->password, $user->U_Password)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid credentials'
            ], 401);
        }

        if ($user->U_Status != 1) {
            return response()->json([
                'status' => 'error',
                'message' => 'Account inactive'
            ], 403);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful',
            'data' => $user->load('role')
        ], 200);

    } catch (Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}


public function sendOtp(Request $request)
{
    try {

        // 🔹 Validation
        $validator = Validator::make($request->all(), [
            'login' => 'required' // email OR mobile
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'validation_error',
                'errors' => $validator->errors()
            ], 422);
        }

        // 🔹 User find by email OR mobile
        $user = UserMaster::where('U_Email', $request->login)
            ->orWhere('U_Mobile', $request->login)
            ->first();

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not found'
            ], 404);
        }

        // 🔹 Generate OTP
        $otp = rand(100000, 999999);

        // 🔹 Save OTP
        $user->OTP_Code = $otp;
        $user->OTP_Expire_At = now()->addMinutes(10);
        $user->save();

        // 🔹 SEND OTP EMAIL
        if (!empty($user->U_Email)) {
            Mail::to($user->U_Email)->send(new OtpMail($otp));
        }

        // 🔹 (Optional) SMS logic yaha add kar sakte ho

        return response()->json([
            'status' => 'success',
            'message' => 'OTP sent successfully'
            // 'otp' => $otp ❌ production me kabhi mat bhejna
        ], 200);

    } catch (Exception $e) {

        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}


public function resetPassword(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'login'       => 'required',
            'otp'         => 'required',
            'new_password'=> 'required|min:6'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'validation_error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = UserMaster::where('U_Email', $request->login)
            ->orWhere('U_Mobile', $request->login)
            ->where('OTP_Code', $request->otp)
            ->where('OTP_Expire_At', '>=', now())
            ->first();

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid or expired OTP'
            ], 400);
        }

        $user->update([
            'U_Password' => Hash::make($request->new_password),
            'OTP_Code' => null,
            'OTP_Expire_At' => null
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Password reset successful'
        ], 200);

    } catch (Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}



}
